// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT license.
import { ProxyInfo, WebsocketConnection, } from "../common.browser/Exports";
import { StringUtils } from "../common/StringUtils";
import { PropertyId } from "../sdk/Exports";
import { ConnectionFactoryBase } from "./ConnectionFactoryBase";
import { WebsocketMessageFormatter, } from "./Exports";
import { HeaderNames } from "./HeaderNames";
import { QueryParameterNames } from "./QueryParameterNames";
export class TranslationConnectionFactory extends ConnectionFactoryBase {
    create(config, authInfo, connectionId) {
        const endpoint = this.getEndpointUrl(config);
        const queryParams = {};
        this.setQueryParams(queryParams, config, endpoint);
        const headers = {};
        if (authInfo.token !== undefined && authInfo.token !== "") {
            headers[authInfo.headerName] = authInfo.token;
        }
        headers[HeaderNames.ConnectionId] = connectionId;
        config.parameters.setProperty(PropertyId.SpeechServiceConnection_Url, endpoint);
        const enableCompression = config.parameters.getProperty("SPEECH-EnableWebsocketCompression", "false") === "true";
        return new WebsocketConnection(endpoint, queryParams, headers, new WebsocketMessageFormatter(), ProxyInfo.fromRecognizerConfig(config), enableCompression, connectionId);
    }
    getEndpointUrl(config, returnRegionPlaceholder) {
        const region = config.parameters.getProperty(PropertyId.SpeechServiceConnection_Region);
        const hostSuffix = ConnectionFactoryBase.getHostSuffix(region);
        let endpointUrl = config.parameters.getProperty(PropertyId.SpeechServiceConnection_Endpoint, undefined);
        if (!endpointUrl) {
            const host = config.parameters.getProperty(PropertyId.SpeechServiceConnection_Host, "wss://{region}.s2s.speech" + hostSuffix);
            endpointUrl = host + "/speech/translation/cognitiveservices/v1";
        }
        if (returnRegionPlaceholder === true) {
            return endpointUrl;
        }
        return StringUtils.formatString(endpointUrl, { region });
    }
    setQueryParams(queryParams, config, endpointUrl) {
        queryParams.from = config.parameters.getProperty(PropertyId.SpeechServiceConnection_RecoLanguage);
        queryParams.to = config.parameters.getProperty(PropertyId.SpeechServiceConnection_TranslationToLanguages);
        this.setCommonUrlParams(config, queryParams, endpointUrl);
        this.setUrlParameter(PropertyId.SpeechServiceResponse_TranslationRequestStablePartialResult, QueryParameterNames.StableTranslation, config, queryParams, endpointUrl);
        const translationVoice = config.parameters.getProperty(PropertyId.SpeechServiceConnection_TranslationVoice, undefined);
        if (translationVoice !== undefined) {
            queryParams.voice = translationVoice;
            queryParams.features = "texttospeech";
        }
    }
}

//# sourceMappingURL=TranslationConnectionFactory.js.map
