/* eslint-disable @typescript-eslint/no-unsafe-assignment */
// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT license.
import * as fs from "fs";
import { LogLevel } from "../sdk/LogLevel";
import { Contracts } from "../sdk/Contracts";
export class ConsoleLoggingListener {
    constructor(logLevelFilter = LogLevel.None) {
        this.privLogPath = undefined;
        this.privEnableConsoleOutput = true;
        this.privLogLevelFilter = logLevelFilter;
    }
    set logPath(path) {
        Contracts.throwIfNullOrUndefined(fs.openSync, "\nFile System access not available");
        this.privLogPath = path;
    }
    set enableConsoleOutput(enableOutput) {
        this.privEnableConsoleOutput = enableOutput;
    }
    onEvent(event) {
        if (event.eventType >= this.privLogLevelFilter) {
            const log = this.toString(event);
            if (!!this.privLogPath) {
                fs.writeFileSync(this.privLogPath, log + "\n", { flag: "a+" });
            }
            if (this.privEnableConsoleOutput) {
                switch (event.eventType) {
                    case LogLevel.Debug:
                        // eslint-disable-next-line no-console
                        console.debug(log);
                        break;
                    case LogLevel.Info:
                        // eslint-disable-next-line no-console
                        console.info(log);
                        break;
                    case LogLevel.Warning:
                        // eslint-disable-next-line no-console
                        console.warn(log);
                        break;
                    case LogLevel.Error:
                        // eslint-disable-next-line no-console
                        console.error(log);
                        break;
                    default:
                        // eslint-disable-next-line no-console
                        console.log(log);
                        break;
                }
            }
        }
    }
    toString(event) {
        const logFragments = [
            `${event.eventTime}`,
            `${event.name}`,
        ];
        const e = event;
        for (const prop in e) {
            if (prop && event.hasOwnProperty(prop) &&
                prop !== "eventTime" && prop !== "eventType" &&
                prop !== "eventId" && prop !== "name" &&
                prop !== "constructor") {
                // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
                const value = e[prop];
                let valueToLog = "<NULL>";
                if (value !== undefined && value !== null) {
                    if (typeof (value) === "number" || typeof (value) === "string") {
                        valueToLog = value.toString();
                    }
                    else {
                        valueToLog = JSON.stringify(value);
                    }
                }
                logFragments.push(`${prop}: ${valueToLog}`);
            }
        }
        return logFragments.join(" | ");
    }
}

//# sourceMappingURL=ConsoleLoggingListener.js.map
